'use strict';

/* --------------------------------------------------------------
 theme_helpers.js 2018-11-13
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2018 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

jse.libs.theme.helpers = jse.libs.theme.helpers || {};

/**
 * Theme Helper Methods
 *
 * This library contains some methods that are required by the theme and need to be defined prior to its
 * initialization. Include this file right after the "initialize_theme.js" and not as a module dependency.
 *
 * Important: If possible, prefer to use the methods of the core JS Engine libraries and not from this library because
 * they can lead to unexpected results or might be hard to use.
 */
(function (exports) {

    'use strict';

    /**
     * Add ":attr" pseudo selector.
     *
     * This pseudo selector is normally enabled by including the JSEngine "jquery_extensions" library. Honeygrid
     * through needs this pseudo selector in this library which might be loeaded prior to jquery_extensions and
     * this is why we define it once again in this file.
     */

    if ($.expr.pseudos.attr === undefined) {
        $.expr.pseudos.attr = $.expr.createPseudo(function (selector) {
            var regexp = new RegExp(selector);
            return function (elem) {
                for (var i = 0; i < elem.attributes.length; i++) {
                    var attr = elem.attributes[i];
                    if (regexp.test(attr.name)) {
                        return true;
                    }
                }
                return false;
            };
        });
    }

    /**
     * Convert all the JS Engine module attributes to the normal state.
     *
     * This method is triggered mostly before a module initialization. Some HTML markup does not have the correct module
     * attribute set because their initialization need to be done in a later time of the page lifecycle.
     *
     * This method will perform the following conversion:
     *
     * ```
     * <!-- Before "setupWidgetAttr" (with the underscore). -->
     * <div data-gambio-_widget="some_widget"></div>
     *
     * ```
     *
     * ```
     * <!-- After "setupWidgetAttr" (the underscore is removed). -->
     * <div data-gambio-widget="some_widget"></div>
     *
     * ```
     *
     * The problem with this method is that the namespaces are hard-coded , the complexity is high and any change in the
     * core JS Engine might break the functionality. Apart from that, the creation and initialization of modules at
     * runtime should be done explicitly by JavaScript modules and HTML markup must not contain such attributes.
     */
    exports.setupWidgetAttr = function ($element) {
        $element.filter(':attr(^data-gx-_), :attr(^data-gambio-_)').add($element.find(':attr(^data-gx-_), :attr(^data-gambio-_)')).each(function () {
            var $self = $(this),
                attributes = $self[0].attributes,
                matchedAttribute,
                namespaceName;

            $.each(attributes, function (index, attribute) {
                if (attribute === undefined) {
                    return true; // wrong attribute, continue loop
                }

                matchedAttribute = attribute.name.match(/data-(gambio|gx)-_.*/g);

                if (matchedAttribute !== null && matchedAttribute.length > 0) {
                    namespaceName = matchedAttribute[0].match(/(gambio|gx)/g)[0];

                    $self.attr(attribute.name.replace('data-' + namespaceName + '-_', 'data-' + namespaceName + '-'), attribute.value);
                }
            });
        });
    };

    /**
     * Fill a form with the provided data.
     *
     * This method will try to fill a form by parsing the provided data. The data have to contain a very specific
     * structure where each value has a "selector" property that points the element to be filled.
     *
     * This method couldn't unfortunately be removed and the use of it should be avoided because it requires that the
     * data generation code must know the selectors and HTML structure of the form, which is a bad practice.
     *
     * @param {object} data Contains the data to be used when filling the form.
     * @param {object} $target jQuery selector for the form or the container of the form to be filled.
     * @param {object} selectorMapping contains the selector mappings of JSON data to the original HTML elements.
     */
    exports.fill = function (data, $target, selectorMapping) {
        $.each(data, function (i, v) {
            if (selectorMapping[v.selector] === undefined) {
                jse.core.debug.warn('The selector mapping "' + v.selector + '" doesn\'t exist.');
                return true;
            }

            var $elements = $target.find(selectorMapping[v.selector]).add($target.filter(selectorMapping[v.selector]));

            $elements.each(function () {
                var $element = $(this);

                switch (v.type) {
                    case 'html':
                        $element.html(v.value);
                        break;
                    case 'attribute':
                        $element.attr(v.key, v.value);
                        break;
                    case 'replace':
                        if (v.value) {
                            $element.replaceWith(v.value);
                        } else {
                            $element.addClass('hidden').empty();
                        }
                        break;
                    default:
                        $element.text(v.value);
                        break;
                }
            });
        });
    };

    /**
     * Get URL parameters the current location or a specific URL.
     *
     * This method was implemented to work with the theme but couldn't unfortunately be replaced with the
     * "getUrlParameters" method inside the "url_arguments" library.
     *
     * If possible, prefer to use the "url_arguments" "getUrlParameters" method instead of this one.
     *
     * @param {string} url (optional) The URL to be parsed, if not provided the current location URL will be used.
     * @param {boolean} deep (optional) Whether to perform a "deep" URL parse.
     *
     * @return {object} Returns an object that contains the parameter values.
     */
    exports.getUrlParams = function (url, deep) {
        url = decodeURIComponent(url || location.href);

        var splitUrl = url.split('?'),
            splitParam = splitUrl.length > 1 ? splitUrl[1].split('&') : [],
            regex = new RegExp(/\[(.*?)\]/g),
            result = {};

        $.each(splitParam, function (i, v) {
            var keyValue = v.split('='),
                regexResult = regex.exec(keyValue[0]),
                base = null,
                basename = keyValue[0].substring(0, keyValue[0].search('\\[')),
                keys = [],
                lastKey = null;

            if (!deep || regexResult === null) {
                result[keyValue[0]] = keyValue[1].split('#')[0];
            } else {

                result[basename] = result[basename] || [];
                base = result[basename];

                do {
                    keys.push(regexResult[1]);
                    regexResult = regex.exec(keyValue[0]);
                } while (regexResult !== null);

                $.each(keys, function (i, v) {
                    var next = keys[i + 1];
                    v = v || '0';

                    if (typeof next === 'string') {
                        base[v] = base[v] || [];
                        base = base[v];
                    } else {
                        base[v] = base[v] || undefined;
                        lastKey = v;
                    }
                });

                if (lastKey !== null) {
                    base[lastKey] = keyValue[1];
                } else {
                    base = keyValue[1];
                }
            }
        });

        return result;
    };
})(jse.libs.theme.helpers);

jse.libs.template = jse.libs.template || {};
jse.libs.template.helpers = jse.libs.theme.helpers;
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
